# Simulations with left- and right-censored data

rm(list=ls(all=TRUE))

#########################################################################
#########################################################################
#########################################################################

# The distance between two visits is a random Exp(lambda) variable. 
  # However, the first visit is not at time 0, which generates left censoring,
  # and there is a maximum follow-up, which generates right censoring.

censor <- function(t, lambda = 1, fu = 12){

  n <- length(t)
  tL <- tR <- NULL
  for(i in 1:n){
    v <- cumsum(rexp(100, lambda)); v[1] <- pmin(v[1], fu*0.99); v <- v[v < fu]; N <- length(v)
    if(t[i] < v[1]){tL[i] <- -Inf; tR[i] <- v[1]}
    else if(t[i] > v[N]){tL[i] <- v[N]; tR[i] <- Inf}
    else{
      w <- which(v > t[i])[1]
      tL[i] <- v[w - 1]; tR[i] <- v[w]
    }
  }
  data.frame(tL = tL, tR = tR, t = t)
}




#########################################################################
#########################################################################
#########################################################################

sim1 <- function(n,lambda, fu){
  x1 <- runif(n)
  x2 <- rexp(n)
  x3 <- rbinom(n, 1, 0.5)

  u <- runif(n)
  t <- (-1*log(1 - u)) + (3*u)*x1 + (3*u^2)*x2 + (3*sqrt(u))*x3
  Y <- censor(t,lambda, fu)
  beta <- list(
   beta0 = function(tau){-1*log(1 - tau)},
   beta1 = function(tau){3*tau},
   beta2 = function(tau){3*tau^2},
   beta3 = function(tau){3*sqrt(tau)}
  )
  list(
    data = data.frame(t = Y$t, tL = Y$tL, tR = Y$tR, x1 = x1, x2 = x2, x3 = x3), 
    formula = formula(Surv(tL,tR, type = "interval2") ~ x1 + x2 + x3), beta = beta
  )
}

#########################################################################
#########################################################################
#########################################################################

# NOTE: as in YNH's examples on breast cancer, I use y = (tL + tR)/2 in DArq(y, x, tL, tR, ...)

simulator <- function(sim, n = 1000, B = 1000, tau = c(0.2,0.4,0.6,0.8), print.each = 50, ...){

  test <- sim(n = 10, ...)
  q <- length(test$beta)
  r <- length(tau)
  true <- lapply(test$beta, function(b) b(tau))

  beta <- matrix(NA, B, r); colnames(beta) <- paste("tau =", tau)
  out <- list(); for(i in 1:q){out[[i]] <- beta}; names(out) <- paste0("beta",0:(q - 1))

  beta_icqr <- beta_YNH <- out
  est_se_icqr <- cover_icqr <- out
  time_icqr <- time_YNH <- NULL

  leftC <- rightC <- NULL

  for(i in 1:B){
    s <- sim(n, ...)
    tL <- s$data$tL; tR <- s$data$tR
    leftC[i] <- mean(tL == -Inf)
    rightC[i] <- mean(tR == Inf)
    X <- s$data[, grep("x", names(s$data))]

    delta <- rep(3,n); delta[tL == -Inf] <- 1; delta[tR == Inf] <- 2
    y <- (tL + tR)/2; y[tL == -Inf] <- tR[tL == -Inf]; y[tR == Inf] <- tL[tR == Inf]
    time_YNH[i] <- system.time(m_YNH <- suppressWarnings(DArq(y, X, delta = delta, tL, tR, taus = tau, iter = 100, tol = 0.01, bootstrap = FALSE)))[3]
    time_icqr[i] <- system.time(m_icqr <- ctqr(s$formula, data = s$data, p = tau))[3]; V <- vcov(m_icqr)

    for(j in 1:q){

      beta_YNH[[j]][i,] <- m_YNH$coef[j,]

      beta.icqr <- m_icqr$coef[j,]
      est.se.icqr <- sapply(V, function(x) sqrt(x[j,j]))
      low <- beta.icqr - 1.96*est.se.icqr
      up <- beta.icqr + 1.96*est.se.icqr

      beta_icqr[[j]][i,] <- beta.icqr
      est_se_icqr[[j]][i,] <- est.se.icqr
      cover_icqr[[j]][i,] <- (low <= true[[j]] & up >= true[[j]])
    }

    # With right-censoring, some large quantiles might be unidentifiable! (maybe)
    conv <- m_icqr$converged
    if(any(alarm <- !conv)){
      for(h in which(alarm)){
        for(j in 1:q){
          beta_YNH[[j]][i,h] <- beta_icqr[[j]][i,h] <- est_se_icqr[[j]][i,h] <- cover_icqr[[j]][i,h] <- NA
        }
      }
    }

    if(round(i/print.each) == i/print.each){print(i)}
  }
  
  outfun <- function(true,beta){
    est <- colMeans(beta, na.rm = TRUE)
    se <- apply(beta,2,sd, na.rm = TRUE)
    mse <- colMeans((beta - t(matrix(true, length(true), nrow(beta))))^2, na.rm = TRUE)
    A <- cbind(true = true, est = est, mse = mse, se = se)
    rownames(A) <- colnames(beta)
    A
  }
  out_icqr <- out_YNH <- list()
  for(j in 1:q){
   out_icqr[[j]] <- outfun(s$beta[[j]](tau), beta_icqr[[j]])
   out_icqr[[j]] <- cbind(out_icqr[[j]], est.se = colMeans(est_se_icqr[[j]], na.rm = TRUE), cover = colMeans(cover_icqr[[j]], na.rm = TRUE))
   out_YNH[[j]] <- outfun(s$beta[[j]](tau), beta_YNH[[j]])
  }
  out <- list(beta_icqr = beta_icqr, beta_YNH = beta_YNH,
    est_se_icqr = est_se_icqr, cover_icqr = cover_icqr, 
    out_icqr = out_icqr, out_YNH = out_YNH, 
    tau = tau, true = s$beta,
    time = data.frame(icqr = time_icqr, YNH = time_YNH), 
    cens = data.frame(leftC = leftC, rightC = rightC)
  )
  class(out) <- "sim"
  out
}

print.sim <- function(x){

  q <- length(x$true)
  for(j in 1:q){
    cat("\n")
    print(paste0("beta", j - 1, ": icqr"))
    cat("\n")
    print(round(x$out_icqr[[j]],2))
    cat("\n")

    print(paste0("beta", j - 1, ": YNH"))
    cat("\n")
    print(round(x$out_YNH[[j]],2))
    cat("\n")

    print("#######################################################")
    cat("\n")
  }
}

# Use var = 0 for the intercept, and var = j for x_j.
# Use what = "beta" for the betas, and what = "se" for the estimated standard errors of "icqr"

plot.sim <- function(obj, which = c("icqr", "YNH"), what = c("beta", "se"), var = 0){

  tau <- obj$tau
  which <- which[1]; what <- what[1]

  beta <- (if(which == "icqr") obj$beta_icqr[[var + 1]] else obj$beta_YNH[[var + 1]])
  se <- obj$est_se_icqr[[var + 1]]

  if(what == "beta"){target <- beta; name <- "beta"}
  else{target <- se; name <- "est.se"}


  title <- (if(var == 0) "Intercept" else paste0("x", var))
  for(j in 1:ncol(target)){

   hist(target[,j], main = title, xlab = paste0(name, var, "(",tau[j],") --- ", which), br = 100)

   if(what == "beta"){
     abline(v = obj$true[[var + 1]](tau[j]), col = "red", lwd = 2)
   }
   else{
     abline(v = obj$out_icqr[[var + 1]][j,"se"], col = "red", lwd = 2)
   }
  }
}

#################################################################################################
#################################################################################################
#################################################################################################


library(ctqr)
library(DArq)
library(quantreg)
B <- 1000
tau <- c(0.25, 0.5, 0.75)

#################################################################################################
#################################################################################################
#################################################################################################

# Note 1: the actual time-to-event is typically between 0 and 20.
# Note 2: the first-step is defined by pchreg(formula, breaks = 2/4/6, splinex = NULL)
# lambda = 2, fu = 10 ---> 10% left cens, 6% right cens
# lambda = 1, fu = 10 ---> 18% left cens, 7% right cens

# Simulation 1: the time between visits is an Exp(2) variable
set.seed(1234); S1 <- simulator(sim1, n = 250, B = B, tau = tau, lambda = 2, fu = 10)
set.seed(1234); T1 <- simulator(sim1, n = 500, B = B, tau = tau, lambda = 2, fu = 10)

# Simulation 2: the time between visits is an Exp(1) variable
set.seed(1234); S2 <- simulator(sim1, n = 250, B = B, tau = tau, lambda = 1, fu = 10)
set.seed(1234); T2 <- simulator(sim1, n = 500, B = B, tau = tau, lambda = 1, fu = 10)

save.image("C:\\Users\\Paolo Frumento\\Desktop\\left_and_righ_censored")

S1
T1
S2
T2

#################################################################################################
#################################################################################################
# Some checks ###################################################################################
#################################################################################################
#################################################################################################

s <- T2 # replace S1,T1,S2,T2
par(mfrow = c(1,3))

plot(s, which = "icqr", var = 0)
plot(s, which = "icqr", var = 1)
plot(s, which = "icqr", var = 2)
plot(s, which = "icqr", var = 3)


plot(s, which = "icqr", var = 0, what = "se")
plot(s, which = "icqr", var = 1, what = "se")
plot(s, which = "icqr", var = 2, what = "se")
plot(s, which = "icqr", var = 3, what = "se")




